# NOTIFICATION SYSTEM - USAGE GUIDE

## Overview
Notification system kamili yenye role-based access control. Kila mtumiaji anaona notifications kulingana na role yake.

## Role-Based Notification Access

### 1. REGISTRAR & REGISTRAR_HQ & PUBLIC_RELATION_OFFICER & SUPERUSER
**Anaona:**
- Notifications za enquiries alizosajili tu (where `registered_by` = user_id)
- Status changes: assigned, rejected, approved, paid, completed

### 2. ASSISTANT_MANAGER
**Anaona:**
- Payments za status = 'initiated'
- Loan applications za status = 'processed'

### 3. GENERAL_MANAGER
**Anaona:**
- Kila kitu (all notifications)

### 4. BRANCH_MANAGER
**Anaona:**
- Notifications za tawi lake tu
- All statuses for his branch

### 5. LOAN_OFFICER
**Anaona:**
- Enquiries zilizo assigned kwake tu
- Notifications za enquiries amepewa tu

### 6. ACCOUNTANT
**Anaona:**
- Enquiries assigned kwake
- Payments approved by general manager
- Loan applications approved

---

## How to Create Notifications

### Example 1: Status Change Notification
```php
use App\Http\Controllers\NotificationController;

// When enquiry status changes
NotificationController::createNotification(
    $enquiryId,
    'status_changed',
    'Enquiry Status Updated',
    "Enquiry {$enquiry->check_number} status changed from {$oldStatus} to {$newStatus}",
    [
        'old_status' => $oldStatus,
        'new_status' => $newStatus,
        'changed_by' => auth()->user()->name,
    ]
);
```

### Example 2: Assignment Notification
```php
// When enquiry is assigned to loan officer
NotificationController::createNotification(
    $enquiryId,
    'enquiry_assigned',
    'New Enquiry Assigned',
    "Enquiry {$enquiry->check_number} has been assigned to you",
    [
        'assigned_by' => auth()->user()->name,
        'enquiry_type' => $enquiry->type,
    ]
);
```

### Example 3: Payment Notification
```php
// When payment is initiated
NotificationController::createNotification(
    $enquiryId,
    'payment_initiated',
    'Payment Initiated',
    "Payment of Tsh {$amount} has been initiated for enquiry {$checkNumber}",
    [
        'amount' => $amount,
        'payment_id' => $payment->id,
        'initiated_by' => auth()->user()->name,
    ]
);

// When payment is approved
NotificationController::createNotification(
    $enquiryId,
    'payment_approved',
    'Payment Approved',
    "Payment of Tsh {$amount} has been approved",
    [
        'amount' => $amount,
        'approved_by' => auth()->user()->name,
    ]
);
```

### Example 4: Loan Application Notification
```php
// When loan is processed
NotificationController::createNotification(
    $enquiryId,
    'loan_processed',
    'Loan Application Processed',
    "Loan application for Tsh {$amount} has been processed",
    [
        'loan_amount' => $amount,
        'loan_type' => $loanType,
        'processed_by' => auth()->user()->name,
    ]
);

// When loan is approved
NotificationController::createNotification(
    $enquiryId,
    'loan_approved',
    'Loan Application Approved',
    "Loan application for Tsh {$amount} has been approved",
    [
        'loan_amount' => $amount,
        'approved_by' => auth()->user()->name,
    ]
);
```

---

## Integration Examples

### In EnquiryController - When Creating New Enquiry
```php
public function store(Request $request)
{
    // Create enquiry
    $enquiry = Enquiry::create($validatedData);

    // Create notification
    NotificationController::createNotification(
        $enquiry->id,
        'enquiry_registered',
        'New Enquiry Registered',
        "Enquiry {$enquiry->check_number} registered for {$enquiry->full_name}",
        [
            'enquiry_type' => $enquiry->type,
            'registered_by' => auth()->user()->name,
        ]
    );

    return redirect()->back()->with('success', 'Enquiry registered successfully');
}
```

### In EnquiryController - When Assigning Enquiry
```php
public function assignUsersToEnquiry(Request $request, $enquiryId)
{
    $enquiry = Enquiry::findOrFail($enquiryId);

    // Assign users
    $enquiry->assignedUsers()->attach($request->user_ids);

    // Update status
    $enquiry->update(['status' => 'assigned']);

    // Create notification
    NotificationController::createNotification(
        $enquiryId,
        'enquiry_assigned',
        'Enquiry Assigned',
        "Enquiry {$enquiry->check_number} has been assigned",
        [
            'assigned_by' => auth()->user()->name,
            'assigned_to_count' => count($request->user_ids),
        ]
    );

    return redirect()->back()->with('success', 'Users assigned successfully');
}
```

### In PaymentController - When Initiating Payment
```php
public function initiate(Request $request, $enquiryId)
{
    $payment = Payment::create([
        'enquiry_id' => $enquiryId,
        'amount' => $request->amount,
        'status' => 'initiated',
        'initiated_by' => auth()->id(),
    ]);

    // Create notification
    NotificationController::createNotification(
        $enquiryId,
        'payment_initiated',
        'Payment Initiated',
        "Payment of Tsh " . number_format($request->amount) . " has been initiated",
        [
            'amount' => $request->amount,
            'payment_id' => $payment->id,
        ]
    );

    return redirect()->back()->with('success', 'Payment initiated successfully');
}
```

### In PaymentController - When Approving Payment
```php
public function approve($paymentId)
{
    $payment = Payment::findOrFail($paymentId);
    $payment->update([
        'status' => 'approved',
        'approved_by' => auth()->id(),
        'approved_at' => now(),
    ]);

    // Create notification
    NotificationController::createNotification(
        $payment->enquiry_id,
        'payment_approved',
        'Payment Approved',
        "Payment of Tsh " . number_format($payment->amount) . " has been approved",
        [
            'amount' => $payment->amount,
            'approved_by' => auth()->user()->name,
        ]
    );

    return redirect()->back()->with('success', 'Payment approved successfully');
}
```

### In LoanController - When Processing Loan
```php
public function process($loanApplicationId)
{
    $loanApplication = LoanApplication::findOrFail($loanApplicationId);
    $loanApplication->update(['status' => 'processed']);

    // Create notification
    NotificationController::createNotification(
        $loanApplication->enquiry_id,
        'loan_processed',
        'Loan Application Processed',
        "Loan application has been processed",
        [
            'loan_amount' => $loanApplication->loan_amount,
            'loan_type' => $loanApplication->loan_type,
        ]
    );

    return redirect()->back()->with('success', 'Loan processed successfully');
}
```

---

## Notification Types

Available notification types:
- `enquiry_registered` - New enquiry registered
- `enquiry_assigned` - Enquiry assigned to user
- `status_changed` - Enquiry status changed
- `payment_initiated` - Payment initiated
- `payment_approved` - Payment approved by manager
- `payment_rejected` - Payment rejected
- `payment_paid` - Payment completed
- `loan_processed` - Loan application processed
- `loan_approved` - Loan approved
- `loan_rejected` - Loan rejected

---

## Routes Available

```php
GET    /notifications                    - View all notifications (with filters)
GET    /notifications/{id}               - View specific notification
POST   /notifications/{id}/mark-read     - Mark as read
GET    /notifications/mark-all-read      - Mark all as read
GET    /notifications-unread-count       - Get unread count (for AJAX)
GET    /notifications-recent             - Get recent 20 (for AJAX)
DELETE /notifications/{id}               - Delete notification
```

---

## Views Available

1. **Header Dropdown** - Shows latest 20 notifications
   - Located in `resources/views/body/header.blade.php`
   - Automatically filtered by role
   - Shows unread count badge

2. **Index Page** - Full list with pagination
   - Located in `resources/views/notifications/index.blade.php`
   - Filters: status, type
   - Pagination: 20 per page

3. **Show Page** - Detailed notification view
   - Located in `resources/views/notifications/show.blade.php`
   - Shows full enquiry details
   - Shows payment/loan details if available

---

## Testing the System

1. **Create a test notification:**
```php
// In tinker or controller
NotificationController::createNotification(
    1, // enquiry_id
    'enquiry_registered',
    'Test Notification',
    'This is a test notification',
    ['test' => true]
);
```

2. **Check notifications:**
   - Navigate to `/notifications`
   - Check header bell icon for count

---

## Next Steps to Fully Integrate

1. Add notification triggers in ALL relevant places:
   - EnquiryController::store() - when creating enquiry
   - EnquiryController::assignUsersToEnquiry() - when assigning
   - PaymentController::initiate() - when initiating payment
   - PaymentController::approve() - when approving payment
   - LoanController::process() - when processing loan
   - Any status change methods

2. Test with different roles to verify access control

3. Add real-time notifications (optional):
   - Use Laravel Echo + Pusher
   - Or implement polling with AJAX

---

## Migration Status

✅ Migration completed successfully
✅ Routes added
✅ Views created
✅ Controller logic implemented
✅ Header integration complete

**Database Table:** `notifications`
**Columns:**
- id, user_id, enquiry_id, type, title, message, meta_data, is_read, read_at, created_at, updated_at
